
#include "DrawManager.h"
#include "Config.h"


// draw attack behavior
void DrawManager::drawAttack(const BWAPI::Unit& unit, const BWAPI::Unit& target, bool re)
{
	if (!Config::Debug::DrawAttack)
		return; 

	BWAPI::Color c;
	if (re)
	{
		c = BWAPI::Colors::Red;
	}
	else
	{
		c = BWAPI::Colors::Blue;
	}

	BWAPI::Broodwar->registerEvent([unit, target, c](BWAPI::Game*){ 
		if (!unit || !unit->exists() || !target || !target->exists())
			return; 
		BWAPI::Broodwar->drawLineMap(unit->getPosition().x, unit->getPosition().y,
		target->getPosition().x, target->getPosition().y, c); 
	},   // action
		nullptr,    // condition
		Config::UnitProbeInterface::FrequencyFrame);  // frames to run

}

void DrawManager::drawMove(const BWAPI::Unit& unit, const BWAPI::Position& p)
{
	if (Config::Debug::DrawMove)
	{
		BWAPI::Broodwar->registerEvent([unit, p](BWAPI::Game*)
		{
			if (!unit || !unit->exists())
				return; 
			BWAPI::Broodwar->drawLineMap(unit->getPosition(), p, BWAPI::Colors::Green);
		},
			nullptr,  // condition
			Config::UnitProbeInterface::FrequencyFrame);  // frames to run
	}

}

void DrawManager::drawRightClick(const BWAPI::Unit& unit, const BWAPI::Unit& target)
{
	if (Config::Debug::DrawRightClick)
	{
		BWAPI::Broodwar->registerEvent([unit, target](BWAPI::Game*)
		{
			if (!unit || !unit->exists() || !target || !target->exists())
				return;
			BWAPI::Broodwar->drawLineMap(unit->getPosition(), target->getPosition(), BWAPI::Colors::Orange);
		},
			nullptr,  // condition
			Config::UnitProbeInterface::FrequencyFrame);  // frames to run
	}

}

void DrawManager::drawBlocksRewards(const BWAPI::Unit& unit, const std::vector<std::vector<double>>& rewards)
{
	if (Config::Debug::DrawBlocksRewards)
	{
		BWAPI::Broodwar->registerEvent([unit, rewards](BWAPI::Game*)
		{
			if (!unit || !unit->exists())
				return;

			int radius = Config::UnitProbeInterface::BlockRadius;
			int span = Config::UnitProbeInterface::BlockSpan;
			BWAPI::Position p = unit->getPosition();

			for (int i = -radius; i <= radius; i++)
			{
				BWAPI::Broodwar->drawLineMap(p.x + span*i, p.y - span*radius, p.x + span*i, p.y + span*radius, BWAPI::Colors::Grey);
			}
			for (int j = -radius; j <= radius; j++)
			{
				BWAPI::Broodwar->drawLineMap(p.x - span*radius, p.y + span*j, p.x + span*radius, p.y + span*j, BWAPI::Colors::Grey);
			}
			for (int i = -radius; i <= radius; i++)
			{
				for (int j = -radius; j <= radius; j++)
					BWAPI::Broodwar->drawTextMap(p.x + span*i, p.y + span*j, "%.0f", rewards[i + radius][j + radius]);
			}		
		},
			nullptr,  // condition
			Config::UnitProbeInterface::FrequencyFrame);  // frames to run
	}
}


void DrawManager::drawPlanBuilding(const BWAPI::TilePosition& location, const BWAPI::UnitType& unit_type)
{
	if (Config::Debug::DrawPlanBuilding)
	{
		BWAPI::Broodwar->registerEvent([location, unit_type](BWAPI::Game*)
		{
			if (!location.isValid() || !unit_type.isBuilding())
				return;
			BWAPI::Broodwar->drawBoxMap(BWAPI::Position(location), 
				BWAPI::Position(location)+BWAPI::Position(unit_type.tileWidth()*32, unit_type.tileHeight()*32), 
				BWAPI::Colors::White);
			BWAPI::Broodwar->drawTextMap(BWAPI::Position(location), "%s", unit_type.getName().c_str());
		},
			nullptr,  // condition
			Config::BuildingManager::FrequencyFrame);  // frames to run
	}

}


void DrawManager::drawAttackTarget(const BWAPI::Unit& unit, const BWAPI::Position& position)
{
	if (Config::Debug::DrawAttackTarget)
	{
		BWAPI::Broodwar->registerEvent([unit, position](BWAPI::Game*)
		{
			if (unit)
				BWAPI::Broodwar->drawBoxMap(unit->getLeft(), unit->getTop(), unit->getRight(), unit->getBottom(), BWAPI::Colors::Cyan);
			else if (position.isValid())
				BWAPI::Broodwar->drawCircleMap(position, 32, BWAPI::Colors::Cyan);
		},
			nullptr,  // condition
			Config::InformationManager::FrequencyFrame - 1);  // frames to run
	}

}


void DrawManager::drawSearchBuild(const BuildActions& actions)
{
	if (Config::Debug::DrawSearchBuild)
	{
		BWAPI::Broodwar->registerEvent([actions](BWAPI::Game*)
		{
			int display_x = 350, display_y = 20; 
			int row = 12, col_action = 130, col_frame = 70; 

			BWAPI::Broodwar->drawBoxScreen(display_x, display_y, display_x+col_action, display_y + row, BWAPI::Colors::Brown); 
			BWAPI::Broodwar->drawTextScreen(display_x, display_y, "BUILD LIST:"); 

			BWAPI::Broodwar->drawTextScreen(display_x + col_action, display_y, "CURRENT:"); 
			BWAPI::Broodwar->drawTextScreen(display_x + col_action, display_y + row, "%d", BWAPI::Broodwar->getFrameCount());

			int num = actions.size(); 
			BWAPI::Broodwar->drawBoxScreen(display_x, display_y + row, display_x + col_action, display_y + (num + 1)*row, BWAPI::Colors::Brown);

			for (int i = 0; i < actions.size(); i++)
			{
				BWAPI::Broodwar->drawTextScreen(display_x, display_y + (i + 1)*row, "%s", actions[i].first.getName().c_str()); 
			}

		},
			nullptr,  // condition
			Config::StrategyManager::FrequencyFrame - 1);  // frames to run
	}
}

void DrawManager::drawTime(int x, int y, int frames, std::string str, double micro_sec)
{
	if (Config::Debug::DrawTime)
	{
		BWAPI::Broodwar->registerEvent([x, y, frames, str, micro_sec](BWAPI::Game*)
		{
			BWAPI::Broodwar->drawTextScreen(x, y, "%s use: %.0f micro sec", str.c_str(), micro_sec);
		},
			nullptr,  // condition
			frames - 1);  // frames to run
	}
}


void DrawManager::drawTerrainData()
{
	//we will iterate through all the base locations, and draw their outlines.
	for (const auto& baseLocation : BWTA::getBaseLocations()) {
		BWAPI::TilePosition p = baseLocation->getTilePosition();

		//draw outline of center location
		BWAPI::Position leftTop(p.x * TILE_SIZE, p.y * TILE_SIZE);
		BWAPI::Position rightBottom(leftTop.x + 4 * TILE_SIZE, leftTop.y + 3 * TILE_SIZE);
		BWAPI::Broodwar->drawBoxMap(leftTop, rightBottom, BWAPI::Colors::Blue);

		//draw a circle at each mineral patch
		for (const auto& mineral : baseLocation->getStaticMinerals()) {
			BWAPI::Broodwar->drawCircleMap(mineral->getInitialPosition(), 30, BWAPI::Colors::Cyan);
		}

		//draw the outlines of Vespene geysers
		for (const auto& geyser : baseLocation->getGeysers()) {
			BWAPI::TilePosition p1 = geyser->getInitialTilePosition();
			BWAPI::Position leftTop1(p1.x * TILE_SIZE, p1.y * TILE_SIZE);
			BWAPI::Position rightBottom1(leftTop1.x + 4 * TILE_SIZE, leftTop1.y + 2 * TILE_SIZE);
			BWAPI::Broodwar->drawBoxMap(leftTop1, rightBottom1, BWAPI::Colors::Orange);
		}

		//if this is an island expansion, draw a yellow circle around the base location
		if (baseLocation->isIsland()) {
			BWAPI::Broodwar->drawCircleMap(baseLocation->getPosition(), 80, BWAPI::Colors::Yellow);
		}
	}

	//we will iterate through all the regions and ...
	for (const auto& region : BWTA::getRegions()) {
		// draw the polygon outline of it in green
		BWTA::Polygon p = region->getPolygon();
		for (size_t j = 0; j < p.size(); ++j) {
			BWAPI::Position point1 = p[j];
			BWAPI::Position point2 = p[(j + 1) % p.size()];
			BWAPI::Broodwar->drawLineMap(point1, point2, BWAPI::Colors::Green);
		}
		// visualize the chokepoints with red lines
		for (auto const& chokepoint : region->getChokepoints()) {
			BWAPI::Position point1 = chokepoint->getSides().first;
			BWAPI::Position point2 = chokepoint->getSides().second;
			BWAPI::Broodwar->drawLineMap(point1, point2, BWAPI::Colors::Red);
		}
	}

	// draw tile index
/*	for (int x = 0; x < BWAPI::Broodwar->mapWidth(); x++)
	{
		for (int y = 0; y < BWAPI::Broodwar->mapHeight(); y++)
			BWAPI::Broodwar->drawTextMap(BWAPI::Position(x*32, y*32), "%d,%d", x, y); 
	}  */
}

/** draw health bar and bullets */
void DrawManager::drawExtendedInterface()
{
	BWAPI::Broodwar->drawTextScreen(300, 20, "%s", "WithoutRetreat");

	int verticalOffset = -10;

	// draw enemy units
	for (auto & ui : BWAPI::Broodwar->enemy()->getUnits())
	{
		BWAPI::UnitType type(ui->getType());
		int hitPoints = ui->getHitPoints();
		int shields = ui->getShields();

		const BWAPI::Position & pos = ui->getPosition();

		int left = pos.x - type.dimensionLeft();
		int right = pos.x + type.dimensionRight();
		int top = pos.y - type.dimensionUp();
		int bottom = pos.y + type.dimensionDown();

		if (!BWAPI::Broodwar->isVisible(BWAPI::TilePosition(ui->getPosition())))
		{
			BWAPI::Broodwar->drawBoxMap(BWAPI::Position(left, top), BWAPI::Position(right, bottom), BWAPI::Colors::Grey, false);
			BWAPI::Broodwar->drawTextMap(BWAPI::Position(left + 3, top + 4), "%s", type.getName().c_str());
		}

		if (!type.isResourceContainer() && type.maxHitPoints() > 0)
		{
			double hpRatio = (double)hitPoints / (double)type.maxHitPoints();

			BWAPI::Color hpColor = BWAPI::Colors::Green;
			if (hpRatio < 0.66) hpColor = BWAPI::Colors::Orange;
			if (hpRatio < 0.33) hpColor = BWAPI::Colors::Red;

			int ratioRight = left + (int)((right - left) * hpRatio);
			int hpTop = top + verticalOffset;
			int hpBottom = top + 4 + verticalOffset;

			BWAPI::Broodwar->drawBoxMap(BWAPI::Position(left, hpTop), BWAPI::Position(right, hpBottom), BWAPI::Colors::Grey, true);
			BWAPI::Broodwar->drawBoxMap(BWAPI::Position(left, hpTop), BWAPI::Position(ratioRight, hpBottom), hpColor, true);
			BWAPI::Broodwar->drawBoxMap(BWAPI::Position(left, hpTop), BWAPI::Position(right, hpBottom), BWAPI::Colors::Black, false);

			int ticWidth = 3;

			for (int i(left); i < right - 1; i += ticWidth)
			{
				BWAPI::Broodwar->drawLineMap(BWAPI::Position(i, hpTop), BWAPI::Position(i, hpBottom), BWAPI::Colors::Black);
			}
		}

		if (!type.isResourceContainer() && type.maxShields() > 0)
		{
			double shieldRatio = (double)shields / (double)type.maxShields();

			int ratioRight = left + (int)((right - left) * shieldRatio);
			int hpTop = top - 3 + verticalOffset;
			int hpBottom = top + 1 + verticalOffset;

			BWAPI::Broodwar->drawBoxMap(BWAPI::Position(left, hpTop), BWAPI::Position(right, hpBottom), BWAPI::Colors::Grey, true);
			BWAPI::Broodwar->drawBoxMap(BWAPI::Position(left, hpTop), BWAPI::Position(ratioRight, hpBottom), BWAPI::Colors::Blue, true);
			BWAPI::Broodwar->drawBoxMap(BWAPI::Position(left, hpTop), BWAPI::Position(right, hpBottom), BWAPI::Colors::Black, false);

			int ticWidth = 3;

			for (int i(left); i < right - 1; i += ticWidth)
			{
				BWAPI::Broodwar->drawLineMap(BWAPI::Position(i, hpTop), BWAPI::Position(i, hpBottom), BWAPI::Colors::Black);
			}
		}

	}

	// draw neutral units and our units
	for (auto & unit : BWAPI::Broodwar->getAllUnits())
	{
		if (unit->getPlayer() == BWAPI::Broodwar->enemy())
		{
			continue;
		}

		const BWAPI::Position & pos = unit->getPosition();

		int left = pos.x - unit->getType().dimensionLeft();
		int right = pos.x + unit->getType().dimensionRight();
		int top = pos.y - unit->getType().dimensionUp();
		int bottom = pos.y + unit->getType().dimensionDown();

		//BWAPI::Broodwar->drawBoxMap(BWAPI::Position(left, top), BWAPI::Position(right, bottom), BWAPI::Colors::Grey, false);

		if (!unit->getType().isResourceContainer() && unit->getType().maxHitPoints() > 0)
		{
			double hpRatio = (double)unit->getHitPoints() / (double)unit->getType().maxHitPoints();

			BWAPI::Color hpColor = BWAPI::Colors::Green;
			if (hpRatio < 0.66) hpColor = BWAPI::Colors::Orange;
			if (hpRatio < 0.33) hpColor = BWAPI::Colors::Red;

			int ratioRight = left + (int)((right - left) * hpRatio);
			int hpTop = top + verticalOffset;
			int hpBottom = top + 4 + verticalOffset;

			BWAPI::Broodwar->drawBoxMap(BWAPI::Position(left, hpTop), BWAPI::Position(right, hpBottom), BWAPI::Colors::Grey, true);
			BWAPI::Broodwar->drawBoxMap(BWAPI::Position(left, hpTop), BWAPI::Position(ratioRight, hpBottom), hpColor, true);
			BWAPI::Broodwar->drawBoxMap(BWAPI::Position(left, hpTop), BWAPI::Position(right, hpBottom), BWAPI::Colors::Black, false);

			int ticWidth = 3;

			for (int i(left); i < right - 1; i += ticWidth)
			{
				BWAPI::Broodwar->drawLineMap(BWAPI::Position(i, hpTop), BWAPI::Position(i, hpBottom), BWAPI::Colors::Black);
			}
		}

		if (!unit->getType().isResourceContainer() && unit->getType().maxShields() > 0)
		{
			double shieldRatio = (double)unit->getShields() / (double)unit->getType().maxShields();

			int ratioRight = left + (int)((right - left) * shieldRatio);
			int hpTop = top - 3 + verticalOffset;
			int hpBottom = top + 1 + verticalOffset;

			BWAPI::Broodwar->drawBoxMap(BWAPI::Position(left, hpTop), BWAPI::Position(right, hpBottom), BWAPI::Colors::Grey, true);
			BWAPI::Broodwar->drawBoxMap(BWAPI::Position(left, hpTop), BWAPI::Position(ratioRight, hpBottom), BWAPI::Colors::Blue, true);
			BWAPI::Broodwar->drawBoxMap(BWAPI::Position(left, hpTop), BWAPI::Position(right, hpBottom), BWAPI::Colors::Black, false);

			int ticWidth = 3;

			for (int i(left); i < right - 1; i += ticWidth)
			{
				BWAPI::Broodwar->drawLineMap(BWAPI::Position(i, hpTop), BWAPI::Position(i, hpBottom), BWAPI::Colors::Black);
			}
		}

		if (unit->getType().isResourceContainer() && unit->getInitialResources() > 0)
		{

			double mineralRatio = (double)unit->getResources() / (double)unit->getInitialResources();

			int ratioRight = left + (int)((right - left) * mineralRatio);
			int hpTop = top + verticalOffset;
			int hpBottom = top + 4 + verticalOffset;

			BWAPI::Broodwar->drawBoxMap(BWAPI::Position(left, hpTop), BWAPI::Position(right, hpBottom), BWAPI::Colors::Grey, true);
			BWAPI::Broodwar->drawBoxMap(BWAPI::Position(left, hpTop), BWAPI::Position(ratioRight, hpBottom), BWAPI::Colors::Cyan, true);
			BWAPI::Broodwar->drawBoxMap(BWAPI::Position(left, hpTop), BWAPI::Position(right, hpBottom), BWAPI::Colors::Black, false);

			int ticWidth = 3;

			for (int i(left); i < right - 1; i += ticWidth)
			{
				BWAPI::Broodwar->drawLineMap(BWAPI::Position(i, hpTop), BWAPI::Position(i, hpBottom), BWAPI::Colors::Black);
			}
		}
	}

	// draw bullets
	for (auto& b : BWAPI::Broodwar->getBullets())
	{
		if (!b->exists())
			continue;
		if (b->getSource() == nullptr || !b->getSource()->exists())
			continue;
		if (b->getTarget() == nullptr || !b->getTarget()->exists())
			continue;
		if (b->getSource()->getPlayer() == BWAPI::Broodwar->self())
			BWAPI::Broodwar->drawLineMap(b->getSource()->getPosition(), b->getTarget()->getPosition(), BWAPI::Colors::Orange);
		else if (b->getSource()->getPlayer() == BWAPI::Broodwar->enemy())
			BWAPI::Broodwar->drawLineMap(b->getSource()->getPosition(), b->getTarget()->getPosition(), BWAPI::Colors::Green);
	}

	// draw target position or unit
	for (auto& ou : BWAPI::Broodwar->self()->getUnits())
	{
		if (ou->getOrderTarget() != nullptr)
			BWAPI::Broodwar->drawLineMap(ou->getPosition(), ou->getOrderTarget()->getPosition(), BWAPI::Colors::Red);
		else if (ou->getOrderTargetPosition().isValid() && ou->getOrderTargetPosition().x>0 && ou->getOrderTargetPosition().y>0)
			BWAPI::Broodwar->drawLineMap(ou->getPosition(), ou->getOrderTargetPosition(), BWAPI::Colors::White);
	}
}